import patterns
import mixer
import device
import transport
import arrangement
import general
import launchMapPages
import playlist
import ui
import channels

import midi
import utils
import time

# Consts

AsparionD700Page_Volume = 0
AsparionD700Page_Pan = 1
AsparionD700Page_Stereo = 2
AsparionD700Page_Channels = 3
AsparionD700Page_Sends = 4
AsparionD700Page_FX = 5 # on selected channel
AsparionD700Page_EQ = 6 # on selected channel

AsparionD700TotalPages = 3

AsparionD700STextCharactersPerRow = 96
AsparionD700STextCharactersPerRowThirdLine = 64
AsparionD700STextCharactersPerDisplay = 12
AsparionD700STextCharactersPerDisplayThirdLine = 8
AsparionD700STextRows = 2

MidiMsgNoteOnLightOn = 0x7F0090
MidiMsgNoteOnLightOff = 0x000090
MidiMsgNoteOn = 0x90
MidiMsgChannelPressure = 0xD0
MidiMsgCC = 0xB0
MidiMsgPitchBend = 0xE0

KnobModeOff = 0
KnobModeNormal = 1
KnobModePan = 2

II_Absolute = 0

# https://www.image-line.com/fl-studio-learning/fl-studio-online-manual/html/midi_scripting.htm

#
# Settings
#
# Users can adjust these settings to their liking, set "False" or "True"

SelectFirstTrackAtBankChange = False
ScrollToFirstTrackAtBankChange = False
SelectMultipleChannelsWithFaderTouch = False
SelectMultipleChannelsWithSelectButton = False


class TAsparionTrack:
    def __init__(self):
        self.TrackNum = 0
        self.TrackNumDisplayed = 0
        self.BaseEventID = 0
        self.KnobEventID = 0
        self.KnobPressEventID = 0
        self.KnobResetEventID = 0
        self.KnobResetValue = 0
        self.KnobMode = 0
        self.KnobValue = 0
        self.KnobCenter = 0
        self.SliderEventID = 0
        self.Tag = 0
        self.SliderName = ""
        self.KnobName = ""
        self.Dirty = False
        self.DirtyColor = False
        self.KnobHeld = False
        self.TextLine1 = ""
        self.TextLine2 = ""
        self.TextLine3 = ""
        self.TextLine1NeedsUpdate = False
        self.TextLine2NeedsUpdate = False
        self.TextLine3NeedsUpdate = False
        self.TextTemp = ""
        self.Name = ""
        self.PotName = ""
        self.TextValuePre = ""
        self.TextValue = ""
        self.Value = 0
        self.LastSentFaderPos = -1
        self.Red = 0
        self.Green = 0
        self.Blue = 0
        self.LastMeterUpdateTime = 0
        self.LastMeterUpdateValue = 0
        self.PeakLeft = 0
        self.PeakRight = 0
        self.PeakClipActiveLeft = 0
        self.PeakClipActiveRight = 0
        self.LastPeakClipActiveLeft = 0
        self.LastPeakClipActiveRight = 0


    def SetTrackColor(self, red, green, blue):

        if ((self.Red != red) | (self.Green != green) | (self.Blue != blue)):
            self.DirtyColor = True   

        self.Red = red
        self.Green = green
        self.Blue = blue


class TAsparionD700:
    def __init__(self):
        self.TempMsgT = ["", ""]
        self.LastTimeMsg = bytearray(10)
        self.Shift = False
        self.TempMsgDirty = False
        self.TempMsgCount = 0
        self.FirstTrack = 0
        self.FirstTrackT = [0, 0]
        self.Tracks = []
        self.Clicking = False
        self.Scrub = False
        self.Flip = False
        self.Page = AsparionD700Page_Volume
        self.PageStartTrack = 0
        self.SmoothSpeed = 0
        self.MeterMax = 0xD
        self.ActivityMax = 0xD
        self.ExNo = 0
        self.TrackOffset = 0
        self.IsMain = False
        self.SelectedTrackNo = 0
        self.SelectedTrackNoGlobal = 0
        self.AsparionD700_ExtenderPosT = ("left", "right")
        self.FreeEventID = 400
        self.AlphaTrack_SliderMax = round(13072 * 16000 / 12800)
        self.GlobalRingPositions = []
        self.GlobalRingEventIDs = []
        self.AdditionalReceiverCount = 0
        self.MasterMute = False


    def OnInit(self, ExNo):

        if not device.isAssigned():
            return

        self.ExNo = ExNo
        self.IsMain = ExNo == 0
        self.TrackOffset = ExNo * 8 + 1
        self.FirstTrackT[0] = self.TrackOffset

        self.Tracks = [0 for x in range(9)]
        for x in range(0, 9):
            self.Tracks[x] = TAsparionTrack()

        self.GlobalRingPositions = [0 for x in range(9)]
        for x in range(0, 9):
            self.GlobalRingPositions[x] = 0

        self.GlobalRingEventIDs = [0 for x in range(9)]
        for x in range(0, 9):
            self.GlobalRingEventIDs[x] = 0
        
        self.FirstTrack = 0
        self.SmoothSpeed = 469
        self.Clicking = True

        device.setHasMeters()
        self.LastTimeMsg = bytearray(10)

        self.InitMeter()
        self.SetPage(self.Page)
        self.UpdateFader(8) # master vol
        self.PrintText("OnInit finished")


    def PrintText(self, text):

        print(time.strftime("%H:%M:%S") + ": " + text)


    def OnDeInit(self):

        self.TurnEverythingOff()
        self.PrintText("OnDeInit finished")


    def TriggerStartupScreen(self):
        if not self.IsMain:
            return

        sysex = bytearray([0xF0, 0x00, 0x00, 0x66, 0x14, 0x41, 0xF7])
        device.midiOutSysex(bytes(sysex))


    def OnDirtyMixerTrack(self, SetTrackNum):

        for m in range(0, len(self.Tracks)):
            if (self.Tracks[m].TrackNum == SetTrackNum) | (SetTrackNum == -1):
                self.Tracks[m].Dirty = True


    def OnRefresh(self, flags):

        if flags & midi.HW_Dirty_Mixer_Sel:
            self.UpdateSelectedTrack()
            
        if flags & midi.HW_Dirty_Mixer_Controls:
            self.UpdateSelectedTrack()
            
        if flags & midi.HW_Dirty_Mixer_Display or flags & midi.HW_Dirty_Mixer_Controls:
            self.UpdateTracks()
            

        # LEDs
        if flags & midi.HW_Dirty_LEDs:
            self.UpdateTracks()
            self.UpdateTransportLEDs()
            

        if flags & midi.HW_Dirty_RemoteLinks: 
            self.UpdateLinkedControllers()

        if  flags & midi.HW_Dirty_RemoteLinkValues:
            self.UpdateLinkedControllers()


    def OnMidiMsg(self, event):

        self.AdditionalReceiverCount = device.dispatchReceiverCount()

        if event.midiId == midi.MIDI_PROGRAMCHANGE:  
            if (self.IsMain):
                pass
            elif event.data1 == 0x44:
                self.SetPage(event.data2)
            elif event.data1 == 0x45:
                self.PageStartTrack = event.data2
            elif event.data1 == 0x55:
                self.SetFirstTrack(self.FirstTrackT[self.FirstTrack] + event.data2)
            elif event.data1 == 0x56:
                self.SetFirstTrack(self.FirstTrackT[self.FirstTrack] - event.data2)
            
        elif event.midiId == midi.MIDI_CONTROLCHANGE:
            if event.midiChan == 0:
                event.inEv = event.data2
                if event.inEv >= 0x40:
                    event.outEv = -(event.inEv - 0x40)
                else:
                    event.outEv = event.inEv

                # knobs
                if event.data1 in [0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17]:
                    r = utils.KnobAccelToRes2(event.outEv)
                    Res = r * (1 / (40 * 2.5))
                    self.SetKnobValue(event.data1 - 0x10, event.outEv, Res)
                    event.handled = True
                else:
                    event.handled = False  # for extra CCs in emulators
            else:
                event.handled = False  # for extra CCs in emulators

        elif event.midiId == midi.MIDI_PITCHBEND:  # pitch bend (faders)
            
            if event.midiChan <= 8:
                event.inEv = event.data1 + (event.data2 << 7)
                event.outEv = (event.inEv << 16) // 16383
                event.inEv -= 0x2000

                if self.Tracks[event.midiChan].SliderEventID >= 0:
                    # slider (mixer track volume)
                    event.handled = True
                    mixer.automateEvent(
                        self.Tracks[event.midiChan].SliderEventID,
                        self.AlphaTrack_SliderToLevel(event.inEv + 0x2000),
                        midi.REC_MIDIController,
                        self.SmoothSpeed,
                    )                                  

        elif event.midiId == midi.MIDI_NOTEON:

            if event.pmeFlags & midi.PME_System != 0:

                event.handled = True

                if (event.data1 == 0x2E) | (event.data1 == 0x2F):  # mixer bank, step 8
                    if event.data2 > 0:
                        incByTracks = (self.AdditionalReceiverCount + 1) * 8
                        self.SetFirstTrack(self.FirstTrackT[self.FirstTrack] + incByTracks * (1 if event.data1 == 0x2F else -1))

                        if self.IsMain:
                            for n in range(0, self.AdditionalReceiverCount):
                                device.dispatch(n, midi.MIDI_PROGRAMCHANGE + ((0x55 if event.data1 == 0x2F else 0x56) << 8) + (incByTracks << 16))

                elif (event.data1 == 0x30) | (event.data1 == 0x31): # channel bank, step 1
                    if event.data2 > 0:
                        incByTracks = 1
                        self.SetFirstTrack(self.FirstTrackT[self.FirstTrack] - 1 + int(event.data1 == 0x31) * 2)

                        if self.IsMain:
                            for n in range(0, self.AdditionalReceiverCount):
                                device.dispatch(n, midi.MIDI_PROGRAMCHANGE + ((0x55 if event.data1 == 0x31 else 0x56) << 8) + (incByTracks << 16))

                elif event.data1 in [0x68, 0x69, 0x6A, 0x6B, 0x6C, 0x6D, 0x6E, 0x6F]: # fader touch
                    if event.data2 > 0:
                        mixer.setTrackNumber(self.Tracks[event.data1- 0x68].TrackNum, midi.curfxMinimalLatencyUpdate | (midi.curfxNoDeselectAll if SelectMultipleChannelsWithFaderTouch else 0))

                elif event.data1 in [0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27]: # knob reset
                    if event.data2 > 0:
                        n = event.data1 - 0x20
                        if self.Page == AsparionD700Page_Sends:
                            if mixer.setRouteTo(mixer.trackNumber(), n + self.PageStartTrack, -1) < 0:
                                self.OnSendTempMsg('Cannot send to this track')
                            else:
                                mixer.afterRoutingChanged()
                        else:
                            self.SetKnobValue(n, midi.MaxInt)

                elif event.data1 == 0x2A: # Pan
                    if event.data2 > 0:
                        if self.Page == AsparionD700Page_Pan:
                            self.SetPage(AsparionD700Page_Stereo)
                        elif self.Page == AsparionD700Page_Stereo:
                            self.SetPage(AsparionD700Page_Volume)
                        else:
                            self.SetPage(AsparionD700Page_Pan)

                elif event.data1 == 0x2C: # EQ
                    if event.data2 > 0:
                        if self.Page == AsparionD700Page_EQ:
                            self.SetPage(AsparionD700Page_Volume)
                        else:
                            self.SetPage(AsparionD700Page_EQ)

                elif event.data1 == 0x29: # Sends
                    if event.data2 > 0:
                        if self.Page == AsparionD700Page_Sends:
                            totalTracks = mixer.trackCount()

                            if (self.PageStartTrack + self.AdditionalReceiverCount * 8 + 8) > totalTracks:
                                self.SetPage(AsparionD700Page_Volume)
                            else:
                                self.PageStartTrack += self.AdditionalReceiverCount * 8 + 8
                                self.SetPage(AsparionD700Page_Sends) # for refresh
                        else:
                            self.PageStartTrack = 0
                            self.SetPage(AsparionD700Page_Sends)

                elif event.data1 == 0x2B: # FX
                    if event.data2 > 0:
                        if self.Page == AsparionD700Page_FX:
                            self.SetPage(AsparionD700Page_Volume)
                        else:
                            self.SetPage(AsparionD700Page_FX)
        
                elif event.data1 == 0x59:  # Metronome
                    if event.data2 > 0:
                        transport.globalTransport(midi.FPT_Metronome, 1, event.pmeFlags)

                elif event.data1 == 0x58:  # Precount metronome, double?
                    if event.data2 > 0:
                        transport.globalTransport(midi.FPT_CountDown, 1, event.pmeFlags)

                elif (event.data1 == 0x5B) | (event.data1 == 0x5C):  # << >>
                    if self.Shift:
                        if event.data2 == 0:
                            v2 = 1
                        elif event.data1 == 0x5B:
                            v2 = 0.5
                        else:
                            v2 = 2
                        transport.setPlaybackSpeed(v2)
                    else:
                        transport.globalTransport(midi.FPT_Rewind + int(event.data1 == 0x5C), int(event.data2 > 0) * 2, event.pmeFlags)
                    device.directFeedback(event)

                elif event.data1 == 0x5D:  # stop
                    transport.globalTransport(midi.FPT_Stop, int(event.data2 > 0) * 2, event.pmeFlags)
                elif event.data1 == 0x5E:  # play
                    transport.globalTransport(midi.FPT_Play, int(event.data2 > 0) * 2, event.pmeFlags)
                elif event.data1 == 0x5F:  # record
                    transport.globalTransport(midi.FPT_Record, int(event.data2 > 0) * 2, event.pmeFlags)

                elif event.data1 == 0x56:  # song/loop changed from 0x5A to 0x56
                    transport.globalTransport(midi.FPT_Loop, int(event.data2 > 0) * 2, event.pmeFlags)

                elif event.data1 == 0x52: # loop double
                    transport.globalTransport(midi.FPT_Snap, int(event.data2 > 0) * 2, event.pmeFlags)

                elif event.data1 == 0x36:  # Magic
                    if event.data2 > 0:
                        transport.globalTransport(midi.FPT_F9, 1)

                elif event.pmeFlags & midi.PME_System_Safe != 0:

                    if (event.data1 >= 0x18) & (event.data1 <= 0x1F):  # select mixer track
                        if event.data2 > 0:
                            i = event.data1 - 0x18
                            ui.showWindow(midi.widMixer)
                            mixer.setTrackNumber(self.Tracks[i].TrackNum, midi.curfxScrollToMakeVisible | midi.curfxMinimalLatencyUpdate | (midi.curfxNoDeselectAll if SelectMultipleChannelsWithSelectButton else 0))

                    elif (event.data1 >= 0x8) & (event.data1 <= 0xF):  # solo
                        if event.data2 > 0:
                            i = event.data1 - 0x8
                            self.Tracks[i].solomode = midi.fxSoloModeWithDestTracks
                            mixer.soloTrack(self.Tracks[i].TrackNum, midi.fxSoloToggle, self.Tracks[i].solomode)
                            mixer.setTrackNumber(self.Tracks[i].TrackNum, midi.curfxScrollToMakeVisible)

                    elif (event.data1 >= 0x10) & (event.data1 <= 0x17):  # mute
                        if event.data2 > 0:
                            mixer.enableTrack(self.Tracks[event.data1 - 0x10].TrackNum)

                    elif (event.data1 == 0x38):  # mute master
                        if event.data2 > 0:
                            mixer.enableTrack(0)

                    elif (event.data1 >= 0x0) & (event.data1 <= 0x7):  # arm
                        if event.data2 > 0:
                            mixer.armTrack(self.Tracks[event.data1].TrackNum)
                    else:
                        event.handled = False
                else:
                    event.handled = False

        elif event.midiId == midi.MIDI_NOTEOFF:
            event.handled = False
        else:
            event.handled = False


    def WriteDisplayPerTrack(self, txt, Row, TrackNo):
        txt = txt.encode("ascii", "ignore").decode() # remove non ascii
        txt = txt.replace(" ", "")
        textLength = len(txt)
        displayWidth = AsparionD700STextCharactersPerDisplay
        startPos = 0

        if (Row == 2):
            displayWidth = AsparionD700STextCharactersPerDisplayThirdLine

        startPos = startPos + TrackNo * displayWidth

        txtEnding = ""

        if (textLength > 8):           
            if txt[textLength - 3].isnumeric():
                txtEnding += txt[textLength - 3]
            if txt[textLength - 2].isnumeric():
                txtEnding += txt[textLength - 2]
            if txt[textLength - 1].isnumeric():
                txtEnding += txt[textLength - 1]

        txt = txt[:displayWidth - len(txtEnding)]
        txt += txtEnding
        txt = txt.ljust(displayWidth, " ")
        
        sysex = bytearray([0xF0, 0x00, 0x00, 0x66, 0x14, 0x1A, startPos, Row + 1])

        if (Row == 0):
            if (self.Tracks[TrackNo].TextLine1 == txt): # only update if changed
                return

            self.Tracks[TrackNo].TextLine1 = txt
        elif (Row == 1):
            if (self.Tracks[TrackNo].TextLine2 == txt): # only update if changed
                return

            self.Tracks[TrackNo].TextLine2 = txt
        elif (Row == 2):
            if (self.Tracks[TrackNo].TextLine3 == txt): # only update if changed
                return

            self.Tracks[TrackNo].TextLine3 = txt

            sysex = bytearray([0xF0, 0x00, 0x00, 0x66, 0x14, 0x19, startPos])
        else:
            return

        sysex += bytearray(txt, "ascii")
        sysex.append(0xF7)
        device.midiOutSysex(bytes(sysex))


    def TurnEverythingOff(self):
        if not device.isAssigned():
            return

        for no in range(0, 7):
            self.UpdateTactLED(0x00 + no, False)
            self.UpdateTactLED(0x08 + no, False)
            self.UpdateTactLED(0x10 + no, False)
            self.UpdateTactLED(0x18 + no, False)

        if self.IsMain:
            # stop
            self.UpdateTactLED(0x5D, False)
            # loop
            self.UpdateTactLED(0x56, False)
            # record
            self.UpdateTactLED(0x5F, False)
            # changed flag
            self.UpdateTactLED(0x50, False)
            # metronome
            self.UpdateTactLED(0x59, False)
            # rec precount
            self.UpdateTactLED(0x58, False)
            # smoothing
            self.UpdateTactLED(0x33, False)
            # Magic
            self.UpdateTactLED(0x36, False)

            # internal turn everything off
            sysex = bytearray([0xF0, 0x00, 0x00, 0x66, 0x14, 0x62])
            sysex.append(0xF7)
            device.midiOutSysex(bytes(sysex))


    def ClearDispalyWithSpaces(self):
        self.WriteDisplayDirty("", 0)
        self.WriteDisplayDirty("", 1)


    def WriteDisplayDirty(self, txt, Row):
        txt = txt.encode("ascii", "ignore").decode() # remove non ascii
        txt = txt[:AsparionD700STextCharactersPerRow].ljust(AsparionD700STextCharactersPerRow, " ")
        sysex = bytearray([0xF0, 0x00, 0x00, 0x66, 0x14, 0x18, AsparionD700STextCharactersPerRow * Row]) + bytearray(txt, "ascii")
        sysex.append(0xF7)
        device.midiOutSysex(bytes(sysex))


    def OnSendTempMsg(self, Msg, Duration=1000):
        pass


    def OnUpdateBeatIndicator(self, Value):
        
        # Blink play button
        SyncLEDMsg = [
            midi.MIDI_NOTEON + (0x5E << 8),
            midi.MIDI_NOTEON + (0x5E << 8) + (0x7F << 16),
            midi.MIDI_NOTEON + (0x5E << 8) + (0x7F << 16),
        ]

        if device.isAssigned():
            device.midiOutNewMsg(SyncLEDMsg[Value], 128)


    def ShowPeakIndicatior(self, chan, isRight, visible):
        
        self.SendMidi(MidiMsgChannelPressure, (chan << 4) | (0xE if visible else 0xF), -1, (0x1 if isRight else 0x0))

    def InitMeter(self):

        if device.isAssigned():
            # clear peak indicators
            for m in range(0, len(self.Tracks) - 1):
                self.ShowPeakIndicatior(m, False, False)
                self.ShowPeakIndicatior(m, True, False)


    def SetPage(self, Value):

        self.Page = Value

        #self.PrintText("Set page: " +  str(self.AdditionalReceiverCount) + "   Me: " + str(self.ExNo))

        if self.IsMain:
            self.UpdateTransportLEDs()
            self.AdditionalReceiverCount = device.dispatchReceiverCount()

            if self.AdditionalReceiverCount != 0:
                for n in range(0, self.AdditionalReceiverCount): # send set page to others
                    device.dispatch(n, midi.MIDI_PROGRAMCHANGE + (0x45 << 8) + ((self.PageStartTrack + 8 + n * 8) << 16))
                    device.dispatch(n, midi.MIDI_PROGRAMCHANGE + (0x44 << 8) + (self.Page << 16))

        self.UpdateTracks()


    def UpdateSelectedTrack(self):

        self.SelectedTrackNoGlobal = mixer.trackNumber()

        if device.isAssigned():
            for m in range(0, len(self.Tracks) - 1):

                if self.Tracks[m].TrackNum == self.SelectedTrackNoGlobal:
                    self.SelectedTrackNo = m

                self.UpdateTactLED(0x18 + m, self.Tracks[m].TrackNum == self.SelectedTrackNoGlobal)

            # if self.Page in [AsparionD700Page_Sends, AsparionD700Page_FX]:
            #    self.UpdateTracks()

            if (self.IsMain):
                self.UpdateLinkedControllers()


    def UpdateLinkedControllers(self):

        if (self.IsMain == False):
            return

        # Update linked controllers A1 to A8
        self.UpdateLinkedController(0x50, 0)
        self.UpdateLinkedController(0x51, 1)
        self.UpdateLinkedController(0x52, 2)
        self.UpdateLinkedController(0x53, 3)

        self.UpdateLinkedController(0x22, 4)
        self.UpdateLinkedController(0x23, 5)
        self.UpdateLinkedController(0x24, 6)
        self.UpdateLinkedController(0x25, 7)


    def UpdateLinkedController(self, ID, No):

        eventID = device.findEventID(midi.EncodeRemoteControlID(device.getPortNumber(), 0, 0) + ID, 1)
        newVal = int(device.getLinkedValue(eventID) * 127)

        if self.GlobalRingPositions[No] != newVal:
            self.GlobalRingPositions[No] = newVal
            device.midiOutMsg(midi.MIDI_CONTROLCHANGE + ((ID) << 8) + (newVal << 16))
        

    def UpdateFader(self, Num):

        data1 = 0
        data2 = 0
        newVal = 0
        newChannel = 0

        if device.isAssigned():

            if Num < 8:

                # V-Pot
                if self.Tracks[Num].KnobEventID >= 0:
                    m = mixer.getEventValue(self.Tracks[Num].KnobEventID, midi.MaxInt, False)
                    newVal = round(int(m / midi.FromMIDI_Max * 127 + 0.49))

                    if self.Tracks[Num].KnobMode == KnobModePan:
                        newChannel = 2
                    elif self.Tracks[Num].KnobMode == KnobModeNormal:
                        newChannel = 1
                    else: # off
                        newChannel = 1

                else:
                    data1 = 0

                if (self.Tracks[Num].KnobValue != newVal):
                    self.Tracks[Num].KnobValue = newVal
                    self.SendMidi(MidiMsgCC, 0x30 + Num, newVal, newChannel)

                # arm, solo, mute

                self.UpdateTactLED(0x00 + Num, int(mixer.isTrackArmed(self.Tracks[Num].TrackNum)) * (1 + int(transport.isRecording())))
                self.UpdateTactLED(0x08 + Num, mixer.isTrackSolo(self.Tracks[Num].TrackNum))
                self.UpdateTactLED(0x10 + Num, not mixer.isTrackEnabled(self.Tracks[Num].TrackNum))
            elif Num == 8: # master
                if (self.MasterMute != (not mixer.isTrackEnabled(0))):
                    self.MasterMute = not mixer.isTrackEnabled(0)
                    self.Tracks[Num].SetTrackColor(255 if self.MasterMute else 0, 0, 0)
                    self.UpdateTrackColor(Num) # master vol click light pos 0x38 0x20+0x18

            # fader
            sv = mixer.getEventValue(self.Tracks[Num].SliderEventID)
            newFaderPos = self.AlphaTrack_LevelToSlider(sv)

            if newFaderPos != self.Tracks[Num].LastSentFaderPos:
                self.Tracks[Num].LastSentFaderPos = newFaderPos
                data1 = newFaderPos
                data2 = data1 & 127
                data1 = data1 >> 7
                self.SendMidi(MidiMsgPitchBend, data2, data1, Num)

            self.Dirty = False


    def AlphaTrack_LevelToSlider(self, Value, Max=midi.FromMIDI_Max):

        return round(Value / Max * self.AlphaTrack_SliderMax)


    def AlphaTrack_SliderToLevel(self, Value, Max=midi.FromMIDI_Max):

        return min(round(Value / self.AlphaTrack_SliderMax * Max), Max)


    def IntToRGB(self, intValue):

        blue = intValue & 0xFF
        green = (intValue & 0xFF00) >> 8
        red = (intValue & 0xFF0000) >> 16
        return (red, green, blue)


    def SendMidi(self, type, data1, data2, channel = 0):

        if (data2 == -1):
           device.midiOutMsg((data1 << 8) | (type + channel))
           return

        device.midiOutMsg((data2 << 16) | (data1 << 8) | (type + channel))


    def UpdateTrackColor(self, m):

        ledID = 0x20 + m
        if (m == 8): # master track
            ledID = 0x38 # # master vol click light pos 0x38

        self.SendMidi(MidiMsgNoteOn, ledID, self.Tracks[m].Red >> 1, 1)
        self.SendMidi(MidiMsgNoteOn, ledID, self.Tracks[m].Green >> 1, 2)
        self.SendMidi(MidiMsgNoteOn, ledID, self.Tracks[m].Blue >> 1, 3)


    def UpdateTracks(self):

        f = self.FirstTrackT[self.FirstTrack]
        CurID = mixer.getTrackPluginId(mixer.trackNumber(), 0)
        sysexTrackNos = bytearray([0xF0, 0x00, 0x00, 0x66, 0x14, 0x17, 0x00])
        
        for m in range(0, len(self.Tracks)):

            self.Tracks[m].KnobPressEventID = -1

            # mixer
            if m == 8:
                self.Tracks[m].TrackNum = -2
                self.Tracks[m].BaseEventID = midi.REC_MainVol
                self.Tracks[m].BaseEventID = mixer.getTrackPluginId(0, 0)
                self.Tracks[m].SliderEventID = (self.Tracks[m].BaseEventID + midi.REC_Mixer_Vol)
                self.Tracks[m].SliderName = "Master Vol"
            else:
                c = mixer.getTrackColor(self.Tracks[m].TrackNum)
                red, green, blue = self.IntToRGB(c)
                self.Tracks[m].SetTrackColor(red, green, blue)
                #self.UpdateTrackColor(m)

                self.Tracks[m].TrackNum = midi.TrackNum_Master + ((f + m) % mixer.trackCount())
                self.Tracks[m].TrackNumDisplayed = self.Tracks[m].TrackNum
                self.Tracks[m].BaseEventID = mixer.getTrackPluginId(self.Tracks[m].TrackNum, 0)
                self.Tracks[m].SliderEventID = (self.Tracks[m].BaseEventID + midi.REC_Mixer_Vol)
                s = mixer.getTrackName(self.Tracks[m].TrackNum)
                self.Tracks[m].SliderName = s + ""
                self.Tracks[m].Name = mixer.getTrackName(self.Tracks[m].TrackNum)                
                self.Tracks[m].KnobEventID = -1
                self.Tracks[m].KnobResetEventID = -1
                self.Tracks[m].KnobResetValue = midi.FromMIDI_Max >> 1
                self.Tracks[m].KnobName = ""
                self.Tracks[m].TextValue = ""
                self.Tracks[m].TextValuePre = ""
                self.Tracks[m].KnobCenter = -1
                self.Tracks[m].KnobMode = KnobModeNormal

                if self.Page == AsparionD700Page_Channels:
                    self.Tracks[m].KnobName = "Channel"
                    self.Name = channels.getChannelName(self.Tracks[m].TrackNum) 
                    self.Tracks[m].Value = int(channels.getChannelVolume(self.Tracks[m].TrackNum) * 100)

                    self.Tracks[m].TextValue = str(self.Tracks[m].Value) + "%"
                    self.Tracks[m].TextValuePre = "Chn"
                    self.WriteDisplayPerTrack(self.Tracks[m].TextValuePre + self.Tracks[m].TextValue, 1, m)
                elif (self.Page == AsparionD700Page_Pan) | (self.Page == AsparionD700Page_Volume):
                    self.Tracks[m].KnobEventID = (self.Tracks[m].BaseEventID + midi.REC_Mixer_Pan)
                    self.Tracks[m].KnobResetEventID = self.Tracks[m].KnobEventID
                    self.Tracks[m].KnobName = ""
                    self.Tracks[m].KnobMode = KnobModePan

                    if self.Page == AsparionD700Page_Volume:
                        self.Tracks[m].TextValuePre = ""
                        val = mixer.getAutoSmoothEventValue(self.Tracks[m].SliderEventID)
                        s = mixer.getEventIDValueString(self.Tracks[m].SliderEventID, val)
                        self.WriteDisplayPerTrack(s, 1, m)
                    else:
                        self.Tracks[m].KnobName = "Pan"
                        self.Tracks[m].Value = mixer.getAutoSmoothEventValue(self.Tracks[m].KnobEventID) // 64 # -6400 to 6400 -> -100 to 100
                        self.Tracks[m].TextValue = str(self.Tracks[m].Value) + "%"
                        self.WriteDisplayPerTrack(self.Tracks[m].TextValuePre + self.Tracks[m].TextValue, 1, m)
                elif self.Page == AsparionD700Page_Stereo:
                    self.Tracks[m].KnobEventID = (self.Tracks[m].BaseEventID + midi.REC_Mixer_SS)
                    self.Tracks[m].KnobResetEventID = self.Tracks[m].KnobEventID
                    self.Tracks[m].KnobName = "Stereo"
                    self.Tracks[m].KnobMode = KnobModePan                   
                    self.Tracks[m].Value = mixer.getAutoSmoothEventValue(self.Tracks[m].KnobEventID) * 100 // 64 # -64 to 64 -> -100 to 100
                    self.Tracks[m].TextValue = str(self.Tracks[m].Value) + "%"
                    self.WriteDisplayPerTrack(self.Tracks[m].TextValuePre + self.Tracks[m].TextValue, 1, m)
                elif self.Page == AsparionD700Page_Sends:
                    
                    self.Tracks[m].TrackNumDisplayed = 0

                    if (mixer.trackCount() <= (self.PageStartTrack + m)):
                        self.Tracks[m].KnobName = "Send"
                        self.Tracks[m].Name = ""
                        self.Tracks[m].TextValue = ""
                        self.WriteDisplayPerTrack(self.Tracks[m].TextValue, 1, m)
                    else:
                        self.Tracks[m].KnobEventID = (CurID + midi.REC_Mixer_Send_First + m + self.PageStartTrack)
                        s = mixer.getEventIDName(self.Tracks[m].KnobEventID) #.replace(" - To ", "")
                        s = s[s.find(' - To '):].replace(" - ", "")
                        self.Tracks[m].KnobResetEventID = self.Tracks[m].KnobEventID
                        self.Tracks[m].KnobName = "Send"
                        self.Tracks[m].KnobResetValue = round(12800 * midi.FromMIDI_Max / 16000)
                        self.Tracks[m].KnobCenter = mixer.getRouteSendActive(mixer.trackNumber(), m + self.PageStartTrack)
                        self.Tracks[m].Name = s
                        self.Tracks[m].Value = mixer.getAutoSmoothEventValue(self.Tracks[m].KnobEventID) // 160 # 0 to 16000 -> 0 to 100
                        self.Tracks[m].TextValue = str(self.Tracks[m].Value) + "%"
                        self.WriteDisplayPerTrack(self.Tracks[m].TextValue, 1, m)
                elif self.Page == AsparionD700Page_FX:
                    self.Tracks[m].TrackNumDisplayed = 0

                    virtualM = m
                    if (self.ExNo == 1):
                        virtualM = 8 + m

                    if ((self.ExNo <= 1) & (virtualM < 10)):

                        CurID = mixer.getTrackPluginId(mixer.trackNumber(), virtualM)
                        self.Tracks[m].KnobEventID = CurID + midi.REC_Plug_MixLevel
                        s = mixer.getEventIDName(self.Tracks[m].KnobEventID)
                        self.Tracks[m].KnobName = "FX"
                        self.Tracks[m].KnobResetValue = midi.FromMIDI_Max
                        self.Tracks[m].Name = s

                        IsValid = mixer.isTrackPluginValid(mixer.trackNumber(), virtualM)
                        IsEnabledAuto = mixer.isTrackAutomationEnabled(mixer.trackNumber(), virtualM)
                        #if IsValid:
                        #    self.Tracks[m].KnobMode = KnobModePan
                        self.Tracks[m].KnobPressEventID = CurID + midi.REC_Plug_Mute
                        #else:
                        self.Tracks[m].KnobMode = KnobModeNormal
                        self.Tracks[m].KnobCenter = int(IsValid & IsEnabledAuto)
                        self.Tracks[m].Value = mixer.getAutoSmoothEventValue(self.Tracks[m].KnobEventID) // 128 # 0 to 16000 -> 0 to 100
                        self.Tracks[m].TextValue = str(self.Tracks[m].Value) + "%"
                        self.Tracks[m].TextValuePre = str(virtualM) + ": "
                    else:
                        self.Tracks[m].Name = ""
                    self.WriteDisplayPerTrack(self.Tracks[m].TextValuePre + self.Tracks[m].TextValue, 1, m)
                elif self.Page == AsparionD700Page_EQ:
                    self.Tracks[m].KnobName = "EQ"
                    self.Tracks[m].TextValuePre = ""
                    self.Tracks[m].KnobEventID = -1
                    self.Tracks[m].KnobMode = KnobModePan
                    self.Tracks[m].TrackNumDisplayed = 0

                    if ((self.ExNo == 0) & (m < 6)):
                        if m < 3:
                            self.Tracks[m].KnobEventID = (CurID + midi.REC_Mixer_EQ_Freq + m)
                            self.Tracks[m].KnobResetEventID = self.Tracks[m].KnobEventID
                            self.Tracks[m].Name = "Freq " + str(m + 1)
                            if (m == 0):
                                self.Tracks[m].KnobResetValue = midi.FromMIDI_Max >> 2
                            elif (m == 1):
                                self.Tracks[m].KnobResetValue = midi.FromMIDI_Max >> 1
                            else:
                                self.Tracks[m].KnobResetValue = midi.FromMIDI_Max
                            self.Tracks[m].KnobCenter = -2
                        elif m < 6:                         
                            self.Tracks[m].KnobEventID = (CurID + midi.REC_Mixer_EQ_Gain + m - 3)
                            self.Tracks[m].KnobResetEventID = self.Tracks[m].KnobEventID
                            self.Tracks[m].Name = "Gain " + str(m - 2)
                            self.Tracks[m].KnobResetValue = midi.FromMIDI_Max >> 1
                            self.Tracks[m].KnobCenter = -2
                    elif (((self.ExNo == 0) & (m < 9)) | ((self.ExNo == 1) & (m == 0))): # Q                          
                        if (m == 0):
                            self.Tracks[m].KnobEventID = (CurID + midi.REC_Mixer_EQ_Q + 2)
                            self.Tracks[m].Name = "Q 3"
                        else:
                            self.Tracks[m].KnobEventID = (CurID + midi.REC_Mixer_EQ_Q + m - 6)
                            self.Tracks[m].Name = "Q " + str(m - 5)

                        self.Tracks[m].KnobResetEventID = self.Tracks[m].KnobEventID
                        self.Tracks[m].KnobResetValue = 17500
                        self.Tracks[m].KnobCenter = -1
                        
                    else:
                        self.Tracks[m].KnobName = ""
                        self.Tracks[m].Name = ""

                    val = mixer.getAutoSmoothEventValue(self.Tracks[m].KnobEventID)
                    s = mixer.getEventIDValueString(self.Tracks[m].KnobEventID, val)
                    self.WriteDisplayPerTrack(self.Tracks[m].TextValuePre + s, 1, m)

                sysexTrackNos.append(self.Tracks[m].TrackNumDisplayed)
                self.WriteDisplayPerTrack(self.Tracks[m].Name, 0, m) # write track names
                self.WriteDisplayPerTrack(self.Tracks[m].KnobName, 2, m) # write track names

            self.UpdateFader(m)

        self.UpdateSelectedTrack()
        
        sysexTrackNos.append(0xF7)
        device.midiOutSysex(bytes(sysexTrackNos))
       

    def SetKnobValue(self, Num, Value, Res=midi.EKRes):

        if (self.Tracks[Num].KnobEventID >= 0) & (self.Tracks[Num].KnobMode < 4):
            if Value == midi.MaxInt:
                if self.Page == AsparionD700Page_FX:
                    if self.Tracks[Num].KnobPressEventID >= 0:

                        Value = channels.incEventValue(self.Tracks[Num].KnobPressEventID, 0, midi.EKRes)
                        channels.processRECEvent(self.Tracks[Num].KnobPressEventID, Value, midi.REC_Controller)
                        s = mixer.getEventIDName(self.Tracks[Num].KnobPressEventID)
                    return
                else:
                    mixer.automateEvent(
                        self.Tracks[Num].KnobResetEventID,
                        self.Tracks[Num].KnobResetValue,
                        midi.REC_MIDIController,
                        self.SmoothSpeed,
                    )
            else:
                mixer.automateEvent(
                    self.Tracks[Num].KnobEventID,
                    Value,
                    midi.REC_Controller,
                    self.SmoothSpeed,
                    1,
                    Res,
                )

            # hint
            n = mixer.getAutoSmoothEventValue(self.Tracks[Num].KnobEventID)
            s = mixer.getEventIDValueString(self.Tracks[Num].KnobEventID, n)
     

    def SetFirstTrack(self, Value):

        self.FirstTrackT[self.FirstTrack] = (Value + mixer.trackCount()) % mixer.trackCount()
        s = utils.Zeros(self.FirstTrackT[self.FirstTrack], 2, " ")

        device.hardwareRefreshMixerTrack(-1)
                                            
        if (self.IsMain and SelectFirstTrackAtBankChange):
            mixer.setTrackNumber(self.FirstTrackT[self.FirstTrack], midi.curfxMinimalLatencyUpdate | (midi.curfxScrollToMakeVisible if ScrollToFirstTrackAtBankChange else 0))


    def OnUpdateMeters(self):

        for m in range(0, len(self.Tracks) - 1):  
            peakL = mixer.getTrackPeaks(self.Tracks[m].TrackNum, midi.PEAK_L)
            peakR = mixer.getTrackPeaks(self.Tracks[m].TrackNum, midi.PEAK_R)

            self.Tracks[m].PeakClipActiveLeft = (peakL > 1)
            self.Tracks[m].PeakClipActiveRight = (peakR > 1)

            self.Tracks[m].PeakLeft = max(self.Tracks[m].PeakLeft, min(round(peakL * self.MeterMax), self.MeterMax))
            self.Tracks[m].PeakRight = max(self.Tracks[m].PeakRight, min(round(peakR * self.MeterMax), self.MeterMax))


    def OnIdle(self):

        # refresh meters
        if device.isAssigned():

            currentTime = time.time()

            for m in range(0, len(self.Tracks) - 1):           
                
                if ((currentTime - self.Tracks[m].LastMeterUpdateTime) < 0.15):
                    continue

                self.Tracks[m].LastMeterUpdateTime = currentTime

                if (self.Tracks[m].PeakRight != 0):
                    self.SendMidi(MidiMsgChannelPressure, (m << 4) | self.Tracks[m].PeakRight, -1, 1)
                    self.Tracks[m].PeakRight = 0

                if (self.Tracks[m].LastPeakClipActiveRight != self.Tracks[m].PeakClipActiveRight):
                    self.ShowPeakIndicatior(m, True, self.Tracks[m].PeakClipActiveRight)
                    self.Tracks[m].LastPeakClipActiveRight = self.Tracks[m].PeakClipActiveRight

                if (self.Tracks[m].PeakLeft != 0):
                    self.SendMidi(MidiMsgChannelPressure, (m << 4) | self.Tracks[m].PeakLeft, -1, 0)
                    self.Tracks[m].PeakLeft = 0                            
                    
                if (self.Tracks[m].LastPeakClipActiveLeft != self.Tracks[m].PeakClipActiveLeft):
                    self.ShowPeakIndicatior(m, False, self.Tracks[m].PeakClipActiveLeft)
                    self.Tracks[m].LastPeakClipActiveLeft = self.Tracks[m].PeakClipActiveLeft
     
                    
        # refresh Colors
        if device.isAssigned():
            for m in range(0, len(self.Tracks) - 1):
                if self.Tracks[m].DirtyColor:
                    self.Tracks[m].DirtyColor = False
                    self.UpdateTrackColor(m)


    def UpdateTactLED(self, ID, isOn):        
        self.SendMidi(MidiMsgNoteOn, ID, (0x7F if isOn else 0x00))


    def UpdateTransportLEDs(self):

        if device.isAssigned() and self.IsMain:
            self.UpdateTactLED(0x2A, (self.Page == AsparionD700Page_Pan) | (self.Page == AsparionD700Page_Stereo))
            self.UpdateTactLED(0x2C, self.Page == AsparionD700Page_EQ)
            self.UpdateTactLED(0x29, self.Page == AsparionD700Page_Sends)
            self.UpdateTactLED(0x2B, self.Page == AsparionD700Page_FX)

            # stop
            self.UpdateTactLED(0x5D, transport.isPlaying() == midi.PM_Stopped)
            # loop
            self.UpdateTactLED(0x56, transport.getLoopMode() == midi.SM_Pat)
            # record
            self.UpdateTactLED(0x5F, transport.isRecording())
            # changed flag
            self.UpdateTactLED(0x50, general.getChangedFlag() > 0)
            # metronome
            self.UpdateTactLED(0x59, general.getUseMetronome())
            # rec precount
            self.UpdateTactLED(0x58, general.getPrecount())
            # smoothing
            self.UpdateTactLED(0x33, self.SmoothSpeed > 0)
            # Magic
            #self.UpdateTactLED(0x32, self.Page != AsparionD700Page_Volume)


    def OnWaitingForInput(self):
        pass


AsparionD700 = TAsparionD700()


def OnInit():
    AsparionD700.OnInit()


def OnDeInit():
    AsparionD700.OnDeInit()


def OnDirtyMixerTrack(SetTrackNum):
    AsparionD700.OnDirtyMixerTrack(SetTrackNum)


def OnRefresh(Flags):
    AsparionD700.OnRefresh(Flags)


def OnMidiMsg(event):
    AsparionD700.OnMidiMsg(event)


def OnSendTempMsg(Msg, Duration=1000):
    AsparionD700.OnSendTempMsg(Msg, Duration)


def OnUpdateBeatIndicator(Value):
    AsparionD700.OnUpdateBeatIndicator(Value)


def OnUpdateMeters():
    AsparionD700.OnUpdateMeters()


def OnIdle():
    AsparionD700.OnIdle()


def OnWaitingForInput():
    AsparionD700.OnWaitingForInput()
